/*********************************************************
*Copyright (C), 2023, Shanghai Eastsoft Microelectronics Co., Ltd.
 * @文件名:    main.c
 * @作  者:    AE Team
 * @版  本:    V1.00
 * @日  期:    2023/02/01
 * @描  述:    RTC演示程序
            MCU与上位机通信，波特率115200。
            复位后发送字符串"Please input time. For example: 2023-01-01 00:00:00"，并等待接收时间信息。
            请按照与“2023-01-01 00:00:00”字符串相同的格式用上位机向MCU发送时间信息。
            然后MCU循环读取实时时间，并打印到上位机显示，周期5s。
            循环打印时间期间仍可通过上位机更改时间信息。
            使用UART_printf函数前需要以下两个步骤：
            1、keil勾选Use MicroLIB 2、IAR/keil #define __PRINTF_USE_UART0__
            RXD - PA22
            TXD - PA23
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "main.h"
#include "rtc.h"

uint8_t g_time_input[19];
uint8_t g_updateflag = 0;

struct tm
{
    uint8_t tm_sec;                   /* Seconds.     [0-60] (1 leap second) */
    uint8_t tm_min;                   /* Minutes.     [0-59] */
    uint8_t tm_hour;                  /* Hours.       [0-23] */
    uint8_t tm_mday;                  /* Day.         [1-31] */
    uint8_t tm_mon;                   /* Month.       [0-11] */
    uint8_t tm_year;                  /* Year - 1900.  */
};

void uart_send(unsigned char *data, unsigned char len)
{
    unsigned char i;

    for (i = 0; i < len; ++i)
    {
        while (!UART_GetFlagStatus(UART0, UART_FLAG_TB));

        UART_SendByte(UART0, data[i]);
    }
}

const char dec_tab[10] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9' };
unsigned char cnt = 0;
unsigned char sec_prev = 0;

struct tm tm;

void uart_puts_time(void)
{
    uint8_t tmstr[64];
    uint8_t i;
    uint8_t *c;

    for (i = 0; i < sizeof(tmstr); ++i)
    {
        tmstr[i] = 0;
    }

    c = tmstr;

    if (tm.tm_sec == sec_prev)
    {
        return;
    }

    sec_prev = tm.tm_sec;

    *c++ = dec_tab[cnt / 100 % 10];
    *c++ = dec_tab[cnt / 10 % 10];
    *c++ = dec_tab[cnt % 10];
    *c++ = ' ';
    *c++ = ' ';
    *c++ = '2';
    *c++ = '0';
    *c++ = dec_tab[((tm.tm_year + 1900) / 10) % 10];
    *c++ = dec_tab[(tm.tm_year + 1900) % 10];
    *c++ = '-';
    *c++ = dec_tab[(tm.tm_mon / 10) % 10];
    *c++ = dec_tab[tm.tm_mon % 10];
    *c++ = '-';
    *c++ = dec_tab[(tm.tm_mday / 10) % 10];
    *c++ = dec_tab[tm.tm_mday % 10];
    *c++ = ' ';
    *c++ = dec_tab[(tm.tm_hour / 10) % 10];
    *c++ = dec_tab[tm.tm_hour % 10];
    *c++ = ':';
    *c++ = dec_tab[(tm.tm_min / 10) % 10];
    *c++ = dec_tab[tm.tm_min % 10];
    *c++ = ':';
    *c++ = dec_tab[(tm.tm_sec / 10) % 10];
    *c++ = dec_tab[tm.tm_sec % 10];
    *c++ = '\n';
    *c = 0;

    cnt++;

    uart_send(tmstr, c - tmstr);
}

int main(void)
{
    uint8_t buf[] = {"Please input time.\nFor example: 2023-01-01 00:00:00\n"};

    HRC_Config(Enable, SCU_HRC_48M, Enable);    /* 配置HRC时钟频率 */
    SystemInit();
    DeviceClockAllEnable();

    User_SysTickInit();
    UARTInit();
    ds1302_init();
    printf("%s", buf);

    while (!g_updateflag); //等待第一次接收到输入的时间

    while (1)
    {
        if (g_updateflag)
        {
            t_time.year = (g_time_input[2] - 0x30) * 10 + (g_time_input[3] - 0x30);
            t_time.month = (g_time_input[5] - 0x30) * 10 + (g_time_input[6] - 0x30);
            t_time.day = (g_time_input[8] - 0x30) * 10 + (g_time_input[9] - 0x30);
            t_time.hour = ((g_time_input[11] - 0x30) * 10 + (g_time_input[12] - 0x30));
            t_time.minute = (g_time_input[14] - 0x30) * 10 + (g_time_input[15] - 0x30);
            t_time.second = (g_time_input[17] - 0x30) * 10 + (g_time_input[18] - 0x30);
            ds1302_write_time();
            g_updateflag = 0;
        }

        ds1302_read_time();

        tm.tm_sec = t_time.second;
        tm.tm_min = t_time.minute;
        tm.tm_hour = t_time.hour;
        tm.tm_mday = t_time.day;
        tm.tm_mon = t_time.month;
        tm.tm_year = t_time.year + 2000 - 1900;

        uart_puts_time();

        Delay_10us(500000);
    }
}
