/*********************************************************
*Copyright (C), 2023, Shanghai Eastsoft Microelectronics Co., Ltd.
 * @文件名:  main.c
 * @作  者:  AE Team
 * @版  本:  V1.00
 * @日  期:  2023/10/17
 * @描  述:  MCU自带Flash模拟EEPROM
 *           开辟ESS Pool。ESS Pool中包含4个ESS Block。每个ESS Block大小为2k字节。ESS Pool的起始地址为0x8000。
 *           测试步骤：
 *           1) 通过ESBurner使能芯片DATAFLASH空间：Page64~Page127；
 *           2) 用ES-LINK2的RX、TX连接开发板的PA22和PA23。
、*              PA22为MCU_UART_RX
、*              PA23为MCU_UART_TX
 *           3) main函数中会调用例子Sample_EES_Control。例子测试完成后，会打印信息。
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          17 Oct. 2023    AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "main.h"

extern void Sample_EES_Control(void);

int fputc(int ch, FILE *f)
{
    uint32_t count = 0;
    FlagStatus status = RESET;

    UART_SendByte(UART0, (unsigned char) ch);

    do
    {
        status = UART_GetFlagStatus(UART0, UART_FLAG_TC);
        count++;
    }
    while ((status == RESET) && (count != 0xFFFF));

    UART_ClearITPendingBit(UART0, UART_FLAG_TC);

    if (count == 0xFFFF)
        return -1;

    return (ch);
}

/*********************************************************
函数名: void UART_Pins_Init(void)
描  述: 初始化UART通讯IO
输入值: 无
输出值: 无
返回值: 无
**********************************************************/
void UART_Pins_Init(void)
{
    GPIO_InitSettingType x;
	memset(&x, 0, sizeof(x));

    x.Signal = GPIO_Pin_Signal_Digital;
    x.Dir = GPIO_Direction_Output;
    x.Func = GPIO_Reuse_Func1;
    x.ODE = GPIO_ODE_Output_Disable;
    x.DS = GPIO_DS_Output_Strong;
    x.PUE = GPIO_PUE_Input_Disable;
    x.PDE = GPIO_PDE_Input_Disable;
    GPIO_Init(GPIO_Pin_A23, &x);        /* TXD - PA23 */

    x.Signal = GPIO_Pin_Signal_Digital;
    x.Func = GPIO_Reuse_Func1;
    x.Dir = GPIO_Direction_Input;
    x.PUE = GPIO_PUE_Input_Disable;
    x.PDE = GPIO_PDE_Input_Disable;
    GPIO_Init(GPIO_Pin_A22, &x);        /* RXD - PA22 */
}


/*********************************************************
函数名: void UARTInit(void)
描  述: UART初始化子程序
输入值: 无
输出值: 无
返回值: 无
**********************************************************/
void UARTInit(void)
{
    UART_InitStruType y;
	memset(&y, 0, sizeof(y));
	
    y.UART_ClockSet = UART_Clock_1;         /* 时钟选择：Pclk */
    y.UART_BaudRate = 115200;               /* 波特率:115200 */
    y.UART_StopBits = UART_StopBits_1;      /* 停止位：1 */
    y.UART_TxMode = UART_DataMode_8;        /* 发送数据格式：8位数据，无校验 */
    y.UART_TxPolar = UART_Polar_Normal;     /* 发送端口极性：正常 */
    y.UART_RxMode = UART_DataMode_8;        /* 接收数据格式：8位数据，无校验 */
    y.UART_RxPolar = UART_Polar_Normal;     /* 接收端口极性：正常 */
    UART_Init(UART0, &y);

    UART_Pins_Init();
    UART0_TxEnable();
    UART0_RxEnable();
}

int main(void)
{
    HRC_Config(Enable, SCU_HRC_48M, Enable);
    SystemInit();
    DeviceClockAllEnable();
    User_SysTickInit();

    SCU_RegUnLock();
    SCU_LVDVS_4V1();        /* LVD阈值4.1V */
    SCU_LVDFLT_Enable();
    SCU_LVDIFS_High();      /* LVDO高电平产生中断（VDD低于阈值产生中断） */
    SCU_LVD_Enable();
    SCU_RegLock();

    __disable_irq();

    /* EEPROM Init */
    while (SCU_GetLVDFlagStatus(SCU_LVDFlag_IF) != RESET);  /* 等到VDD电压高于LVD阈值 */
    
    UARTInit();
    
    Sample_EES_Control();
    
    __enable_irq();

    while (1);
}

/*************************END OF FILE**********************/
